<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Dashboard - Bar VIP Admin</title>
    
    <!-- CSS Principal -->
    <link rel="stylesheet" href="assets/css/admin.css">
    
    <!-- Chart.js para gráficos -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    
    <!-- Font Awesome para iconos -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <!-- Sidebar -->
    <nav class="sidebar" id="sidebar">
        <div class="sidebar-header">
            <div class="logo">
                <i class="fas fa-cocktail"></i>
                <span>Bar VIP Admin</span>
            </div>
            <button class="sidebar-toggle" id="sidebarToggle">
                <i class="fas fa-bars"></i>
            </button>
        </div>
        
        <ul class="sidebar-nav">
            <li class="nav-item active">
                <a href="index.php" class="nav-link">
                    <i class="fas fa-chart-line"></i>
                    <span>Dashboard</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="clientes.php" class="nav-link">
                    <i class="fas fa-users"></i>
                    <span>Clientes</span>
                    <span class="badge" id="clientesBadge">0</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="validaciones.php" class="nav-link">
                    <i class="fas fa-user-check"></i>
                    <span>Validaciones</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="reportes.php" class="nav-link">
                    <i class="fas fa-chart-bar"></i>
                    <span>Reportes</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="imagenes.php" class="nav-link">
                    <i class="fas fa-images"></i>
                    <span>Galería</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="configuracion.php" class="nav-link">
                    <i class="fas fa-cog"></i>
                    <span>Configuración</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="logs.php" class="nav-link">
                    <i class="fas fa-list-alt"></i>
                    <span>Logs</span>
                </a>
            </li>
        </ul>
        
        <div class="sidebar-footer">
            <div class="admin-info">
                <i class="fas fa-user-circle"></i>
                <div>
                    <div class="admin-name" id="adminName">Administrador</div>
                    <div class="admin-role">Super Admin</div>
                </div>
            </div>
            <a href="logout.php" class="logout-btn">
                <i class="fas fa-sign-out-alt"></i>
                <span>Cerrar Sesión</span>
            </a>
        </div>
    </nav>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <!-- Top Bar -->
        <header class="topbar">
            <div class="topbar-left">
                <button class="mobile-toggle" id="mobileToggle">
                    <i class="fas fa-bars"></i>
                </button>
                <div class="breadcrumb">
                    <i class="fas fa-chart-line"></i>
                    <span>Dashboard</span>
                </div>
            </div>
            <div class="topbar-right">
                <div class="search-box">
                    <i class="fas fa-search"></i>
                    <input type="text" placeholder="Buscar cliente..." id="quickSearch">
                </div>
                <button class="refresh-btn" id="refreshData" title="Actualizar datos">
                    <i class="fas fa-sync"></i>
                </button>
                <div class="admin-dropdown">
                    <img src="assets/images/admin-avatar.png" alt="Admin" class="admin-avatar">
                    <i class="fas fa-chevron-down"></i>
                </div>
            </div>
        </header>

        <!-- Dashboard Content -->
        <main class="content">
            <!-- Stats Cards -->
            <div class="stats-grid">
                <div class="stat-card primary">
                    <div class="stat-icon">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-number" id="totalClientes">0</div>
                        <div class="stat-label">Total Clientes</div>
                        <div class="stat-change positive">
                            <i class="fas fa-arrow-up"></i>
                            <span id="clientesChange">0%</span>
                        </div>
                    </div>
                </div>
                
                <div class="stat-card success">
                    <div class="stat-icon">
                        <i class="fas fa-user-plus"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-number" id="registrosHoy">0</div>
                        <div class="stat-label">Registros Hoy</div>
                        <div class="stat-change">
                            <i class="fas fa-calendar-day"></i>
                            <span>Hoy</span>
                        </div>
                    </div>
                </div>
                
                <div class="stat-card info">
                    <div class="stat-icon">
                        <i class="fas fa-user-check"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-number" id="validacionesHoy">0</div>
                        <div class="stat-label">Validaciones Hoy</div>
                        <div class="stat-change positive">
                            <i class="fas fa-percentage"></i>
                            <span id="tasaExito">0%</span>
                        </div>
                    </div>
                </div>
                
                <div class="stat-card warning">
                    <div class="stat-icon">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-number" id="fallosHoy">0</div>
                        <div class="stat-label">Fallos Hoy</div>
                        <div class="stat-change negative">
                            <i class="fas fa-arrow-down"></i>
                            <span id="fallosChange">0%</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Charts Row -->
            <div class="charts-row">
                <div class="chart-card">
                    <div class="chart-header">
                        <h3><i class="fas fa-chart-line"></i> Registros por Día (Últimos 30 días)</h3>
                        <div class="chart-controls">
                            <select id="periodSelector">
                                <option value="7">7 días</option>
                                <option value="30" selected>30 días</option>
                                <option value="90">90 días</option>
                            </select>
                        </div>
                    </div>
                    <canvas id="registrosChart"></canvas>
                </div>
                
                <div class="chart-card">
                    <div class="chart-header">
                        <h3><i class="fas fa-chart-pie"></i> Validaciones por Estado</h3>
                    </div>
                    <canvas id="validacionesChart"></canvas>
                </div>
            </div>

            <!-- Activity Tables -->
            <div class="activity-row">
                <div class="activity-card">
                    <div class="activity-header">
                        <h3><i class="fas fa-clock"></i> Actividad Reciente</h3>
                        <a href="logs.php" class="view-all">Ver todo</a>
                    </div>
                    <div class="activity-list" id="actividadReciente">
                        <!-- Se llena dinámicamente -->
                    </div>
                </div>
                
                <div class="activity-card">
                    <div class="activity-header">
                        <h3><i class="fas fa-user-plus"></i> Últimos Registros</h3>
                        <a href="clientes.php" class="view-all">Ver todos</a>
                    </div>
                    <div class="client-list" id="ultimosRegistros">
                        <!-- Se llena dinámicamente -->
                    </div>
                </div>
            </div>

            <!-- System Status -->
            <div class="system-status">
                <div class="status-card">
                    <div class="status-header">
                        <h3><i class="fas fa-server"></i> Estado del Sistema</h3>
                        <div class="status-indicator online" id="systemStatus"></div>
                    </div>
                    <div class="status-content">
                        <div class="status-item">
                            <span class="status-label">Base de Datos:</span>
                            <span class="status-value connected" id="dbStatus">Conectado</span>
                        </div>
                        <div class="status-item">
                            <span class="status-label">Face-API:</span>
                            <span class="status-value connected" id="faceApiStatus">Disponible</span>
                        </div>
                        <div class="status-item">
                            <span class="status-label">Espacio Usado:</span>
                            <span class="status-value" id="storageUsed">0 MB</span>
                        </div>
                        <div class="status-item">
                            <span class="status-label">Último Backup:</span>
                            <span class="status-value" id="lastBackup">Nunca</span>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Loading Overlay -->
    <div class="loading-overlay" id="loadingOverlay">
        <div class="loading-spinner">
            <i class="fas fa-spinner fa-spin"></i>
            <span>Cargando datos...</span>
        </div>
    </div>

    <!-- Scripts -->
    <script src="assets/js/admin.js"></script>
    <script>
        // Dashboard específico
        class Dashboard {
            constructor() {
                this.charts = {};
                this.refreshInterval = null;
                this.init();
            }

            async init() {
                await this.loadDashboardData();
                this.initCharts();
                this.setupEventListeners();
                this.startAutoRefresh();
            }

            async loadDashboardData() {
                showLoading();
                try {
                    const response = await fetch('api/dashboard.php');
                    const data = await response.json();
                    
                    if (data.success) {
                        this.updateStats(data.stats);
                        this.updateActivity(data.activity);
                        this.updateSystemStatus(data.system);
                        this.updateChartData(data.charts);
                    }
                } catch (error) {
                    console.error('Error loading dashboard:', error);
                    showError('Error cargando dashboard');
                } finally {
                    hideLoading();
                }
            }

            updateStats(stats) {
                document.getElementById('totalClientes').textContent = stats.total_clientes || 0;
                document.getElementById('registrosHoy').textContent = stats.registros_hoy || 0;
                document.getElementById('validacionesHoy').textContent = stats.validaciones_hoy || 0;
                document.getElementById('fallosHoy').textContent = stats.validaciones_fallidas_hoy || 0;
                
                // Calcular tasa de éxito
                const totalValidaciones = stats.validaciones_hoy || 0;
                const exitosas = stats.validaciones_exitosas_hoy || 0;
                const tasaExito = totalValidaciones > 0 ? Math.round((exitosas / totalValidaciones) * 100) : 0;
                document.getElementById('tasaExito').textContent = tasaExito + '%';
                
                // Actualizar badge del sidebar
                document.getElementById('clientesBadge').textContent = stats.total_clientes || 0;
            }

            updateActivity(activity) {
                const activityContainer = document.getElementById('actividadReciente');
                const clientsContainer = document.getElementById('ultimosRegistros');
                
                activityContainer.innerHTML = activity.logs.map(log => `
                    <div class="activity-item">
                        <div class="activity-icon ${log.type}">
                            <i class="fas ${this.getActivityIcon(log.accion)}"></i>
                        </div>
                        <div class="activity-content">
                            <div class="activity-text">${log.descripcion}</div>
                            <div class="activity-time">${this.formatTime(log.timestamp)}</div>
                        </div>
                    </div>
                `).join('');
                
                clientsContainer.innerHTML = activity.clients.map(client => `
                    <div class="client-item">
                        <div class="client-avatar">
                            ${client.selfie_path ? 
                                `<img src="../${client.selfie_path}" alt="${client.nombre}">` : 
                                `<i class="fas fa-user"></i>`
                            }
                        </div>
                        <div class="client-info">
                            <div class="client-name">${client.nombre}</div>
                            <div class="client-time">${this.formatTime(client.creado_en)}</div>
                        </div>
                    </div>
                `).join('');
            }

            updateSystemStatus(system) {
                document.getElementById('dbStatus').textContent = system.db_connected ? 'Conectado' : 'Desconectado';
                document.getElementById('dbStatus').className = system.db_connected ? 'status-value connected' : 'status-value disconnected';
                
                document.getElementById('storageUsed').textContent = system.storage_used + ' MB';
                document.getElementById('lastBackup').textContent = system.last_backup || 'Nunca';
            }

            initCharts() {
                // Gráfico de registros por día
                const registrosCtx = document.getElementById('registrosChart').getContext('2d');
                this.charts.registros = new Chart(registrosCtx, {
                    type: 'line',
                    data: {
                        labels: [],
                        datasets: [{
                            label: 'Registros',
                            data: [],
                            borderColor: '#2563eb',
                            backgroundColor: 'rgba(37, 99, 235, 0.1)',
                            fill: true,
                            tension: 0.4
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                display: false
                            }
                        },
                        scales: {
                            y: {
                                beginAtZero: true,
                                grid: {
                                    color: 'rgba(0,0,0,0.1)'
                                }
                            },
                            x: {
                                grid: {
                                    color: 'rgba(0,0,0,0.1)'
                                }
                            }
                        }
                    }
                });

                // Gráfico de validaciones
                const validacionesCtx = document.getElementById('validacionesChart').getContext('2d');
                this.charts.validaciones = new Chart(validacionesCtx, {
                    type: 'doughnut',
                    data: {
                        labels: ['Exitosas', 'Fallidas', 'Error'],
                        datasets: [{
                            data: [0, 0, 0],
                            backgroundColor: ['#10b981', '#f59e0b', '#ef4444'],
                            borderWidth: 0
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                position: 'bottom'
                            }
                        }
                    }
                });
            }

            updateChartData(charts) {
                if (charts.registros) {
                    this.charts.registros.data.labels = charts.registros.labels;
                    this.charts.registros.data.datasets[0].data = charts.registros.data;
                    this.charts.registros.update();
                }

                if (charts.validaciones) {
                    this.charts.validaciones.data.datasets[0].data = [
                        charts.validaciones.exitosas,
                        charts.validaciones.fallidas,
                        charts.validaciones.error
                    ];
                    this.charts.validaciones.update();
                }
            }

            getActivityIcon(accion) {
                const icons = {
                    'login_admin': 'fa-sign-in-alt',
                    'usuario_creado': 'fa-user-plus',
                    'usuario_actualizado': 'fa-user-edit',
                    'usuario_eliminado': 'fa-user-minus',
                    'validacion_exitosa': 'fa-check',
                    'validacion_fallida': 'fa-times',
                    'backup_creado': 'fa-database'
                };
                return icons[accion] || 'fa-info';
            }

            formatTime(timestamp) {
                const date = new Date(timestamp);
                const now = new Date();
                const diff = now - date;
                
                if (diff < 60000) return 'Hace un momento';
                if (diff < 3600000) return `Hace ${Math.floor(diff/60000)} min`;
                if (diff < 86400000) return `Hace ${Math.floor(diff/3600000)} h`;
                return date.toLocaleDateString('es-ES');
            }

            setupEventListeners() {
                document.getElementById('refreshData').addEventListener('click', () => {
                    this.loadDashboardData();
                });

                document.getElementById('periodSelector').addEventListener('change', (e) => {
                    this.updatePeriod(e.target.value);
                });

                document.getElementById('quickSearch').addEventListener('input', (e) => {
                    this.quickSearch(e.target.value);
                });
            }

            async updatePeriod(days) {
                showLoading();
                try {
                    const response = await fetch(`api/dashboard.php?period=${days}`);
                    const data = await response.json();
                    
                    if (data.success) {
                        this.updateChartData(data.charts);
                    }
                } catch (error) {
                    console.error('Error updating period:', error);
                } finally {
                    hideLoading();
                }
            }

            async quickSearch(query) {
                if (query.length < 2) return;
                
                try {
                    const response = await fetch(`api/search.php?q=${encodeURIComponent(query)}`);
                    const data = await response.json();
                    
                    // Implementar dropdown de resultados
                    console.log('Search results:', data);
                } catch (error) {
                    console.error('Search error:', error);
                }
            }

            startAutoRefresh() {
                this.refreshInterval = setInterval(() => {
                    this.loadDashboardData();
                }, 300000); // 5 minutos
            }

            destroy() {
                if (this.refreshInterval) {
                    clearInterval(this.refreshInterval);
                }
                Object.values(this.charts).forEach(chart => chart.destroy());
            }
        }

        // Inicializar dashboard cuando el DOM esté listo
        document.addEventListener('DOMContentLoaded', () => {
            window.dashboard = new Dashboard();
        });

        // Limpiar al salir
        window.addEventListener('beforeunload', () => {
            if (window.dashboard) {
                window.dashboard.destroy();
            }
        });
    </script>
</body>
</html>